/*!************************************************************************
 * FILE :         clAudioSource.cpp
 * SW-COMPONENT:  Audio Engine
 * DESCRIPTION:
 *
 * Each AudioSource in the system is represented by an instance of AudioSource
 * or by a subclass of AudioSource.
 * AudioSource instances are used and controlled by the AudioSourceController.
 *
 * The individual subclasses of AudioSources implement the source-specific
 * communication with the corresponding MW components.
 *
 * Each AudioSource exposes an interface to the AudioSourceController with
 * the methods:
 * - vOn(...)     start source (player) and start demute-ramp
 * - vOff(...)    start mute-ramp and stop source (player)
 * - vPause(...)  start mute-ramp and pause source (player)
 * - bIsMuted(...)
 * - vReset(...)  stop player and mute source
 *
 * The AudioSource notifies the AudioSourceController via Callback-methods:
 * - vOn_Done()    the demute-ramp was started
 * - vOff_Done()...the mute-ramp has finished, player was stopped
 * - vPause_Done() the mute-ramp has finished, player was paused
 *
 * The AudioSource can have the states: (please see the state-charts for details)
 * - Off
 * - Pause
 * - On        demute-ramp was finished. Source is playing
 * - RampUp    AudioSource has started the demute-ramp
 * - RampDown  AudioSource has started the mute-ramp
 *
 * A statemachine ensures that requests are performed following the state-
 * transition rules.
 *
 * Sequence is:
 * - AudioSourceController calls vOn, vOff or vPause
 * - clAudioSource delegates request to StateMachine
 * - StateMachine applies rule-checks and might invoke vMW_On, vMW_Off, vMW_Pause
 * - vMW_On, vMW_Off, vMW_Pause do the communication with the MW
 *   Individual SubClasses of clAudioSouce may overwrite this methods to
 *   establish a specific MW communcation.
 * - Each implementation of vMW_On, vMW_Off, vMW_Pause has to confirm the operation
 *   asynchronously by calling vMW_OnDone, vMW_OffDone, vMW_PauseDone
 * - vMW_OnDone, vMW_OffDone, vMW_PauseDone will delegate the trigger
 *   again to the StateMachine.
 * - The StateMachine applies again the rules and invokes one of the methods:
 *   vNotifyOnDone,  vNotifyOffDone and  vNotifyPauseDone
 *   in order to inform the AudioSourceController
 *
 * TASKS TO IMPLEMENT A NEW AUDIOSOURCE:
 * - Provide a subclass of clAudioSource
 * - Overwrite the methods: vMW_On, vMW_Off, vMW_Pause
 *   Implement within these methods the specific communication with MW
 * - Ensure that these methods confirm the operation by invoking:
 *   vMW_OnDone, vMW_OffDone, vMW_PauseDone
 *
 *
 * AUTHOR:        CM-DI/PJ-VW34 Steinle
 * COPYRIGHT:     (c) 2006 Blaupunkt Werke
 *************************************************************************/
#if defined PROJECTBUILD_VWLL_NF || defined PROJECTBUILD_NISSAN_LCN || defined PROJECTBUILD_NISSAN_LCN_NAR
  #define OSAL_S_IMPORT_INTERFACE_GENERIC
  #include "osal_if.h"
#endif


#if defined PROJECTBUILD_VWLL_NF || defined PROJECTBUILD_NISSAN_LCN || defined PROJECTBUILD_NISSAN_LCN_NAR
  #include "GUI_Widget/CR_util/src/UTIL_Utf8String.h"
  #include "GUI_Widget/Trace/GUI_Trace.h"
#else
  #include "string.h"
#endif

#ifdef WIN32
#include "windows.h"
#endif

//#include "fiaudio/hsi_audio_defines.h"

//#include "hmi_master/clHMIMngr.h"
#include <stdlib.h>
#include <time.h>
#include <pthread.h>

#include "AudioStack/clAudioSMEngine.h"
#include "AudioStack/clAudioSource.h"
#include "AudioStack/SMT/clSrcStateFactory.h"
#include "AudioStack/SMT/clSrcState.h"

#include "AudioStack/AudioSources.h"
#include "AudioStack/clGeniviAudioCtrlAdapter.h"
#include "AudioStack/AudioSources/clAudioSourceFactory.h"

#ifndef USE_DLT_TRACE
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_COMP_AUDIOSTACK
#include "trcGenProj/Header/clAudioSource.cpp.trc.h"
#endif


const char * AudioSources::SourceName[] = {
#define ADD_AUDIOSOURCE(x) #x ,
      #include "AudioStack/AudioSourcesEnumeration.h"
      #undef ADD_AUDIOSOURCE
};

namespace AudioStack
{

/**
 * Public constructor for instanziating a source that won't do
 * external communications. Could be used as mute source
 */
clAudioSource::clAudioSource(SourceID enSrcId) :
       m_pclState(NULL)
       , m_pacName(NULL)
       , m_SrcId(enSrcId)
       , m_possibleNextSource(enSrcId)
       , m_u32UserData(0)
       , m_u32PlayCmdCounter(0)
       , m_StackRule(enSrcId)
       , m_pclAudioSourceObserver(NULL)
       , m_availability(clAudioSource::not_available)
       , m_availabilityReason(clAudioSource::nomedia)
{
   std::string str("UNDEF");
   if((m_StackRule.getRegistration() == clStackRules::registerNone)
         || (m_StackRule.getRegistration() == clStackRules::registerStatic))
   {
      str.assign(AudioSources::SourceName[enSrcId.enSourceClass]);
      m_availability = clAudioSource::available;
      m_availabilityReason = clAudioSource::samemedia;
   }

   char *cstr = new char[str.length() + 1];
   strcpy(cstr, str.c_str());
   m_pacName = cstr;

   m_pclState = clSrcStateFactory::pclCreateOff();
   m_pclState->vEnter(this, 0, clSrcState::ENTRYCONDITION_INITIAL);
}

/**
 * \brief
 * Constructor
 *
 * \param pacName
 * name represents the audio source using a string
 *
 * \param u8Id
 * u8Id represents the audio source
 *
 * Constructor
 */
clAudioSource::clAudioSource(const tChar *pacName, SourceID SrcId) :
             m_pclState(NULL)
             , m_pacName(NULL)
             , m_SrcId(SrcId)
             , m_possibleNextSource(SrcId)
             , m_u32UserData(0)
             , m_u32PlayCmdCounter(0)
             , m_StackRule(SrcId)
             , m_pclAudioSourceObserver(NULL)
             , m_availability(clAudioSource::not_available)
             , m_availabilityReason(clAudioSource::newmedia)
{
   std::string str(pacName);
   char *cstr = new char[str.length() + 1];
   strcpy(cstr, str.c_str());
   m_pacName = cstr;

   m_pclState = clSrcStateFactory::pclCreateOff();
   m_pclState->vEnter(this, 0, clSrcState::ENTRYCONDITION_INITIAL);
}

tVoid clAudioSource::vMW_CCAResponse(tS32 extSourceId, enSourceActivity activity)
{
   ETG_TRACE_ERR(("ERROR: vMW_CCAResponse called for base class %d, subID %d",
         ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(m_SrcId.enSourceClass)),
         (tU16) m_SrcId.u16SubSource));
}


tVoid clAudioSource::Initialize(const tChar *pacName, SourceID srcId)
{

   /* check the name and set the stack rules */
   /// OBSOLETE DUE TO STACKRULE CONSTRUCTOR
   /// REQUIRES SOURCEID AND SETS CORRECTLY
   /// WITHIN INIT LIST
//   if (!m_StackRule.setRules(this))
//   {
//      ETG_TRACE_ERRMEM(("Could not find Rule for SourceClass: %d",
//            ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(srcId.enSourceClass))));
//      return;
//   }
   // init the state machine
}

/**
* /brief
* Destructor
*/
clAudioSource::~clAudioSource()
{
  m_pclAudioSourceObserver = NULL;
  OSAL_DELETE[] m_pacName;
  m_pclState = NULL;
}

/**
* /brief
* vReset() sets all internal members to an initial state
*/
void clAudioSource::vReset()
{
   // init the state machine
   m_pclState = clSrcStateFactory::pclCreateOff();
   m_pclState->vEnter(this, 0, clSrcState::ENTRYCONDITION_INITIAL);
   m_u32UserData = 0;
   m_possibleNextSource = SourceID(AudioSources::NONE, 0);
   m_u32PlayCmdCounter = 0;
}

void clAudioSource::vPrint()
{
   ETG_TRACE_ERR(("%s"
      , m_pacName));
   ETG_TRACE_ERR(("  | SourceClass: 0x%02x(%03d), SubID 0x%02x(%03d),"
      , m_SrcId.enSourceClass
      , m_SrcId.enSourceClass
      , m_SrcId.u16SubSource
      , m_SrcId.u16SubSource));
   ETG_TRACE_ERR(("  | MidwID 0x%02x(%03d)"
      , m_StackRule.getExternalID()
      , m_StackRule.getExternalID()));
   ETG_TRACE_ERR(("  | State: %d"
      , ETG_CENUM(AudioStates::enAudioStates, m_pclState->m_enStateID)));
   ETG_TRACE_ERR(("  | Availability: %d"
      , ETG_CENUM(clAudioSource::enSourceAvailability, m_availability)));
   ETG_TRACE_ERR(("                "));
}

tBool clAudioSource::bCCACallback(tU32 u32msgcode, tU32 u32msgparam1, tU32 msgparam2)
{
   ETG_TRACE_USR1(("clAudioSource::bCCACallback: Default Implementation"));
   return TRUE;
}

/**
* /brief
* vSets member Variable to given value.
* This is used for RadioPresets or
* NaviAnnouncement Index (ACR FI, ChannelRequest Index)
*/
tVoid clAudioSource::vSetUserData(tU32 u32UserData)
{
   ETG_TRACE_USR4(("vSetUserData:  Data %d SourceClass: %d subID %#x(%d)"
         , u32UserData
         ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(sGetId().enSourceClass))
         , (tU16)sGetId().u16SubSource
         , (tU16)sGetId().u16SubSource));

   m_u32UserData = u32UserData;
}


/**
 * \brief
 * allowOnTop guide the request to the audio stack object.
 *
 * \param src
 * pointer to the audio source to be checked.
 *
 * \returns
 * allowed, disallowed or kicked
 */
clStackRules::allow_t clAudioSource::allowOnTop(clAudioSource* src)
{
   if(src == NULL)
   {
      return clStackRules::disallowed;
   }
  return m_StackRule.allowOnTop(src->m_SrcId.enSourceClass);
}

/**
 * \brief
 * pacGetState
 *
 * \returns
 * returns the name of the current state of the source
 */
const tChar *clAudioSource::pacGetState()
{
   const tChar* retVal = "not initialized";
   if (m_pclState)
   {
      retVal = m_pclState->m_pacName;
   }
   return retVal;
}

/**
 * \brief
 * u8GetId
 *
 * \returns
 * return the id of the audio source
 */
SourceID clAudioSource::sGetId()
{
   return m_SrcId;
}

/**
 * \brief
 * u8GetNameIndex guides the request to the audio stack
 *
 * \returns
 * returns the index of the audio source
 */
//tU8 clAudioSource::u8GetNameIndex()
//{
//  return m_StackRule.getSrcNameIndex();
//}

/**
 * \brief
 * getPushAction guides the request to the audio stack
 *
 * \param index
 * index of the audio source to be checked
 *
 * \returns
 * returns the push actions
 */
clStackRules::actions_t clAudioSource::getPushAction(tU8 indx)
{
  return m_StackRule.getPushAction(indx);
}

/**
 * \brief
 * getPopAction guides the request to the audio stack
 *
 * \param index
 * index of the audio source to be checked
 *
 * \returns
 * returns the pop actions
 */
clStackRules::actions_t clAudioSource::getPopAction(tU8 indx)
{
  return m_StackRule.getPopAction(indx);
}

/**
 * \brief
 * Write brief comment for getType here.
 *
 * \returns
 * returns the type of this audio source (foreground, background, mix, etc.)
 */
clStackRules::type_t clAudioSource::getType()
{
  return m_StackRule.getType();
}

/**
 * \brief
 * Write brief comment for getGroup here.
 *
 * \returns
 * returns the group of this audio source (radio, media, virtual, none)
 */
clStackRules::group_t clAudioSource::getGroup()
{
  return m_StackRule.getGroup();
}

tS32 clAudioSource::getExternalId()
{
   return m_StackRule.getExternalID();
}


tVoid clAudioSource::vWaitFor()
{
  if(NULL != m_pclState)
    {
      m_pclState->vMsg_WaitFor(this, (tU32)0);
    }
}

/*!************************************************************************
 * METHOD:        vOn
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from AudioSourceController
 *                Delegate to StateMachine
 *
 * PARAMETER:     callback    reference to AudioSourceController for callback
 *                u32UserData reserved
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vOn()
{
   if(NULL != m_pclState)
   {
     m_pclState->vMsg_On(this, (tU32)0);
   }
}

/*!************************************************************************
 * METHOD:        vOff
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from AudioSourceController
 *                Delegate to StateMachine
 *
 * PARAMETER:     callback    reference to AudioSourceController for callback
 *                u32UserData reserved
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vOff(SourceID possibleNextSource)
{
   m_possibleNextSource = possibleNextSource;

   if(NULL != m_pclState && bPreCheckStateChange2Off())
   {
      m_pclState->vMsg_Off(this, (tU32)0);
   }
}

/*!************************************************************************
 * METHOD:        vPause
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from AudioSourceController
 *                Delegate to StateMachine
 *
 * PARAMETER:     callback    reference to AudioSourceController for callback
 *                u32UserData reserved
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vPause(SourceID u8PossibleNextSource)
{
   m_possibleNextSource = u8PossibleNextSource;

   if(NULL != m_pclState)
   {
      m_pclState->vMsg_Pause(this, (tU32)0);
   }
}

/*!************************************************************************
 * METHOD:        bIsActive
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from AudioSourceController
 *
 *
 * PARAMETER:
 *
 * RETURNVALUE:   TRUE, if AudioSource is in state: rampUp, On or rampDown
 *                FALSE, if AudioSource is in state: pause or off
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tBool clAudioSource::bIsActive()
{
   if (m_pclState == clSrcStateFactory::pclCreateOff()
      || m_pclState == clSrcStateFactory::pclCreatePause()
      || m_pclState == clSrcStateFactory::pclCreateWaitFor())
   {
      return FALSE;
   }
   return TRUE;
}

/*!************************************************************************
 * METHOD:        vNotifyOnDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                Notify AudioSourceController
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vNotifyOnDone()
{
   if (m_pclAudioSourceObserver)
   {
      m_pclAudioSourceObserver->vOn_done(this);
   }
}

/*!************************************************************************
 * METHOD:        vNotifyOnDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                Notify AudioSourceController
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vNotifyOffDone()
{
   if (m_pclAudioSourceObserver)
   {
      m_pclAudioSourceObserver->vOff_done(this);
   }
}

/*!************************************************************************
 * METHOD:        vNotifyOnDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                Notify AudioSourceController
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vNotifyPauseDone()
{
   if (m_pclAudioSourceObserver)
   {
      m_pclAudioSourceObserver->vPause_done(this);
   }
}

/*!************************************************************************
 * METHOD:        vNotifyInitStarted
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                Notify AudioSourceController
 *                Event is given, when Audiosource enters the Init-State
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vNotifyInitStarted()
{
   if (m_pclAudioSourceObserver)
   {
      m_pclAudioSourceObserver->vInit_started(this);
   }
}

/*!************************************************************************
 * METHOD:        vMW_On
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                This is the default implementation
 *                The default impl. just confirms the operation.
 *                Overwrite this method to establish MW communication
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_On()
{
   vMW_RequestConfirmed();
   vMW_OnDone();
}

/*!************************************************************************
 * METHOD:        vMW_Init
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                This is the default implementation
 *                The default impl. just confirms the operation.
 *                Overwrite this method to establish MW communication
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_Init()
{
   vMW_InitDone();
}

/*!************************************************************************
 * METHOD:        bIsSourceAvailable
 * CLASS:         clAudioSource
 * DESCRIPTION:   Checks if source is ready to get activated,
 *                e.g. CD is inserted, IPod is authenticated...
 *
 * PARAMETER:
 * RETURNVALUE:   TRUE if Player is ready
 *                FALSE if Player is not ready yet
 *
 * AUTHOR:        R.Volkmer
 *************************************************************************/
tVoid clAudioSource::vSourceAvailablilityChange(enSourceAvailability availability,
      enSourceAvailabilityReason availabilityReason)
{
   if((availability == clAudioSource::not_available)
      && (m_availability != clAudioSource::not_available))
   {
      m_availability = availability;
      m_availabilityReason = availabilityReason;
      clAudioSMEngine::Source_Removed(m_SrcId);
      return;
   }

   if((m_availability != clAudioSource::available)
      && (availability == clAudioSource::available)
      && (availabilityReason == clAudioSource::newmedia)
      && m_StackRule.isAutoPlay())
   {
      ETG_TRACE_USR1(("vSourceAvailablilityChange: Trigger AutoPlay SourceClass: %d subID %#x(%d) Available: %d Reason: %d"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(m_SrcId.enSourceClass))
            , (tU16)m_SrcId.u16SubSource
            , (tU16)m_SrcId.u16SubSource
            , ETG_CENUM(clAudioSource::enSourceAvailability, availability)
            , ETG_CENUM(clAudioSource::enSourceAvailabilityReason, availabilityReason)));
      m_availability = availability;
      m_availabilityReason = availabilityReason;
      clAudioSMEngine::Source_Insert(m_SrcId);
      return;
   }

   m_availability = availability;
   m_availabilityReason = availabilityReason;
   if(m_availability == clAudioSource::unkonwn)
   {
      vWatchdogStart_Availability();
   }

   if(m_pclState != NULL)
   {
      m_pclState->vMsg_SrcAvailabilityChg(this, m_u32UserData);
   }
}

tVoid clAudioSource::vSourceAvailablilityChange()
{
   if(m_pclState != NULL)
   {
      m_pclState->vMsg_SrcAvailabilityChg(this, m_u32UserData);
   }
}

/*!************************************************************************
 * METHOD:        enIsSourceAvailable
 * CLASS:         clAudioSource
 * DESCRIPTION:   Checks if source is ready to get activated,
 *                e.g. CD is inserted, IPod is authenticated...
 *
 * PARAMETER:
 * RETURNVALUE:   TRUE if Player is ready
 *                FALSE if Player is not ready yet
 *
 * AUTHOR:        R.Volkmer
 *************************************************************************/
clAudioSource::enSourceAvailability clAudioSource::enIsSourceAvailable()
{
   ETG_TRACE_USR4(("enIsSourceAvailable: SourceClass: %d subID %#x(%d) Available: %d"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(m_SrcId.enSourceClass))
            , (tU16)m_SrcId.u16SubSource
            , (tU16)m_SrcId.u16SubSource
            , ETG_CENUM(clAudioSource::enSourceAvailability, m_availability)));
   return m_availability;
}

/*!************************************************************************
 * METHOD:        vMW_Off
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                This is the default implementation
 *                The default impl. just confirms the operation.
 *                Overwrite this method to establish MW communication
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_Off(SourceID )
{
   vMW_OffDone();
}

/*!************************************************************************
 * METHOD:        vMW_Pause
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                This is the default implementation
 *                The default impl. just confirms the operation.
 *                Overwrite this method to establish MW communication
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_Pause(SourceID )
{
   vMW_PauseDone();
}

/*!************************************************************************
 * METHOD:        vMW_OnDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Confirmation from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_OnDone()
{
   if(NULL != m_pclState)
   {
     ETG_TRACE_USR2(("vMW_OnDone Source: %d, SubSource %d, State: %d"
         ,ETG_CENUM(AudioSources::enAudioSources, static_cast<AudioSources::enAudioSources>(m_SrcId.enSourceClass))
         ,m_SrcId.u16SubSource
         ,ETG_CENUM(AudioStates::enAudioStates, m_pclState->enGetStateID())
         ));
      m_pclState->vMsg_OnDone(this, 0);
   }
}

/*!************************************************************************
 * METHOD:        vMW_OffDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Confirmation from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_OffDone()
{
   if(NULL != m_pclState)
   {
     ETG_TRACE_USR2(("vMW_OffDone Source: %d, SubSource %d, State: %d"
         ,ETG_CENUM(AudioSources::enAudioSources, static_cast<AudioSources::enAudioSources>(m_SrcId.enSourceClass))
         ,m_SrcId.u16SubSource
         ,ETG_CENUM(AudioStates::enAudioStates, m_pclState->enGetStateID())
         ));
      m_pclState->vMsg_OffDone(this, 0);
   }
   if(  (m_StackRule.getRegistration() == clStackRules::registerDynamic)
      &&(m_availability == clAudioSource::not_available))
      {
         ETG_TRACE_USR1(("Remove Source: Source: %d, SubSource %d, State: %d"
            ,ETG_CENUM(AudioSources::enAudioSources, static_cast<AudioSources::enAudioSources>(m_SrcId.enSourceClass))
            ,m_SrcId.u16SubSource
            ,ETG_CENUM(AudioStates::enAudioStates, m_pclState->enGetStateID())
            ));
         clAudioSourceFactory::removeAudioSources(m_SrcId);
      }
}

/*!************************************************************************
 * METHOD:        vMW_InitDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Confirmation from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_InitDone()
{
   if(NULL != m_pclState)
   {
     ETG_TRACE_USR2(("vMW_InitDone Source: %d, SubSource %d, State: %d"
         ,ETG_CENUM(AudioSources::enAudioSources, static_cast<AudioSources::enAudioSources>(m_SrcId.enSourceClass))
         ,m_SrcId.u16SubSource
         ,ETG_CENUM(AudioStates::enAudioStates, m_pclState->enGetStateID())
         ));
     m_pclState->vMsg_InitDone(this, 0);
   }
}

/*!************************************************************************
 * METHOD:        vMW_RequestConfirmed
 * CLASS:         clAudioSource
 * DESCRIPTION:   Confirmation from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_RequestConfirmed()
{
   if(NULL != m_pclState)
   {
      m_pclState->vMsg_RequestConfirmed(this, 0);
   }
}

/*!************************************************************************
 * METHOD:        vMW_PauseDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Confirmation from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_PauseDone()
{
   if(NULL != m_pclState)
   {
     ETG_TRACE_USR2(("vMW_PauseDone Source: %d, SubSource %d, State: %d"
         ,ETG_CENUM(AudioSources::enAudioSources, static_cast<AudioSources::enAudioSources>(m_SrcId.enSourceClass))
         ,m_SrcId.u16SubSource
         ,ETG_CENUM(AudioStates::enAudioStates, m_pclState->enGetStateID())
         ));
      m_pclState->vMsg_PauseDone(this, 0);
   }
}

/*!************************************************************************
 * METHOD:        bIsOff
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from AudioSourceController
 *
 *
 * PARAMETER:
 *
 * RETURNVALUE:   TRUE, if AudioSource is in state: Off
 *                FALSE, otherwise
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tBool clAudioSource::bIsOff()
{
   if ( m_pclState == clSrcStateFactory::pclCreateOff() )
   {
      return TRUE;
   }
   return FALSE;
}

/*!************************************************************************
 * METHOD:        pclGetState
 * CLASS:         clAudioSource
 * DESCRIPTION:   simple getter for prop
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
clSrcState* clAudioSource::pclGetState()
{
   return this->m_pclState;
}

/*!************************************************************************
 * METHOD:        vMW_ServiceUnavailable
 * CLASS:         clAudioSource
 * DESCRIPTION:   Notification from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_ServiceUnavailable()
{
   if(NULL != m_pclState)
   {
      m_pclState->vMsg_ServiceUnavailable(this, 0);
   }
}

/*!************************************************************************
 * METHOD:        vMW_ServiceAvailable
 * CLASS:         clAudioSource
 * DESCRIPTION:   Notification from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_ServiceAvailable()
{
   if(NULL != m_pclState)
   {
      m_pclState->vMsg_ServiceAvailable(this, 0);
   }
}

/*!************************************************************************
 * METHOD:        bPreCheckStateChange2Off
 * CLASS:         clAudioSource
 * DESCRIPTION:   check the cmd counter for play commands. this has to be 0
 *                after decrease, what means, that there is no channel request
 *                open any longer, so the channel can be closed now. Increasing
 *                the counter has to be done by each source itself, which needs
 *                this counter (currently only navi is doing so)
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW37)
 *************************************************************************/
tBool clAudioSource::bPreCheckStateChange2Off()
{
   if (m_u32PlayCmdCounter != 0)
   {
      m_u32PlayCmdCounter--;

      if(m_u32PlayCmdCounter == 0)
         return TRUE;
      else
      {
         ETG_TRACE_USR4(("bPreCheckStateChange2Off(): m_u32PlayCmdCounter: %d", m_u32PlayCmdCounter));
         return FALSE;
      }
   }
   else
      return TRUE; // default when source does not use the cmd counter
}

tVoid clAudioSource::vWatchdogStart_Availability()
{
   ETG_TRACE_USR4(("vStartWatchdog_Availability()"));
   pthread_t thread_id;
   pthread_create(&thread_id, NULL, &clAudioSource::vStart_ThreadedTimer, (void*)this);
}

tVoid* clAudioSource::vStart_ThreadedTimer(void* pAudSrc)
{

   clAudioSource* pAudioSource = (clAudioSource*) pAudSrc;
   if(pAudioSource != NULL)
   {
      ETG_TRACE_USR4(("vStart_ThreadedTimer(): sleep %ds", pAudioSource->getAvailabilityTimeout()));
      sleep(pAudioSource->getAvailabilityTimeout());
      ETG_TRACE_USR4(("vStart_ThreadedTimer(): awoke"));
      clGeniviAudioCtrlAdapter::vSwitchThreadContext<clAudioSource>(pAudioSource,&clAudioSource::vSourceAvailablilityChange);
   }else{
      ETG_TRACE_FATAL(("ERROR: vStart_ThreadedTimer(): pAudioSource == NULL"));
   }
   return NULL;
}



} //namespace


/* EOF */

